;
; Heap management routines.  
; The memory from 0 to HeapTop is allocated and freed
; dynamically through the routines GetHeap and FreeHeap.
; The heap management algorithms used were chosen with
; the assumptions that few requests for heap storage would
; be made, and that allocated heap would not often be freed
; Also it was felt that efficiency in allocating/freeing 
; heap was less important than minimizing the range of
; memory actually used by the heap.  In short, these are 
; not the algorithms to use in a multi-tasking operating system.
; A discussion of the algorithms is in Knuths Fund Alg, Ch 2.5,A,B.
; Alg A (GetHeap) was modified to allocate memory from the beginning
; of a free block.
; The heap free list is a singly linked list of blocks of 
; available RAM in the heap.  The first word in each block
; is the size of the block, the second word points to the next
; free block in the list.  The list is kept sorted in order of
; block address, lowest first.
;
; GetHeap returns the address + 2 (caller must not change size
; word of block) of the first block in the free list (first fit)
; large enough for the caller.  If none is found it attempts
; to satisfy the request by extending the heap.  The attempt fails
; if the high heap address + minimum stack size > stack base.
;
; FreeHeap returns to the free list a block allocated by GetHeap.
; If the returned block is physically adjacent to one or both of
; its neighbors in the free list, the blocks are combined and the
; free list updated accordingly.
;
; The head of the free list is at location 0, and looks like -
; heap head :
;	size	dcw	0	; so it never gets merged.
;	next	struct word	; points to next free block.
;	temp0	struct word	; scratch area for heap routines.
;	temp1	struct word	; ditto.
; The power up routine initializes the heap as follows :
;	addr	val
;	0	0		; head size = 0.
;	2	8		; next free block at 8.
;	8	default heap size - 8 from eeprom.
;	10	-1		; nil next pointer.
;
; Kernel entries :
; GetHeap - acquire semi-permanent storage from low memory.
; FreHeap - free storage from GetHeap.
; ChkHeap - check heap free list pointers and sizes.

; Kernel variables :
; Heaptop - heap memory is from 0..HeapTop.
; StkBas - value of sp when stack is empty.
; StkMin - minimum difference between StkBas and HeapTop.
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; GetHeap :
; Enter with byte count on stack.
; On return, carry set if request couldn't be satisfied,
; else byte count replaced with pointer to allocated memory.
;
; Calling sequence -
;	pea	##nbytes	; push requested buffer size.
;	jsl	>0,GetHeap	; get buffer if possible.
;	bcs	error		; br if not enough heap.
;				; pointer to buffer is on stack.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
getheap:

$size equ 0			; size offset in block.
$next equ 2			; next ptr offset in block.
$temp0 equ 4			; temp area in head block.
$temp1 equ 6			; ditto

	clc			; assume enough heap.
	php
	rep	#0x30
	pha
	phx
	phy
	phd
;	phb

;
; Set dbr = dpr = 0.
; 
;	pea	##0
;	plb
;	phb
;	pld

	lda	##0
	tcd

;
; Add free block overhead (1 word) to size request.
;
	lda	13,s		; get requested size.
	inc	a		; add 2 for block size
	inc	a		; in first word of block.
	sta	13,s		; save size to allocate.
;
; Search for a block in the free list 
; of size >= requested amount.  When found,
; x will point at it and y will point at the
; previous block.
;
	ldx	##0		; point x to list head.
	bra	$nxtfre		; advance to next free block.
$srch:
	lda	<$size,x	; get size of this block.
	sec
	sbc	13,s		; subtract requested amt.
	bcs	$found		; br if result >= 0, use this block.
$nxtfre:
	txy			; point y at current  block.
	ldx	<$next,y	; point x to next free block.
	bpl	$srch		; br unless end of free list.
	brl	$extend		; reached end, try extend heap.
$found:

;
; Found a big enough block.
; x points to the block, y to its predecessor.
; acc = block size - requested size.  If this
; remainder is small, just allocate the whole block
; (don't bother keeping track of tiny blocks), otherwise
; we give them the first part of the block and make a
; new free block with the rest.  
;
	cmp	##18		; Big enough to split ?
	bcs	$split		; br if yes.
;
; Too small to split.  Point the previous block
; at the next block (unlinking this block from list),
; replace stacked request with pointer to block + 2,
; and we're done.
;
	inx			; point x at callers portion
	inx			; of block (and pointer to next).
	txa
	sta	13,s		; save return val for caller.
	lda	<$size,x	; get pointer to next block.
	tyx			; point x at previous block.
	sta	<$next,x	; point prev to next.
	brl	$done		; clean up stack and rtl.

$split:

;
; Oh sigh.  Split the block, first part for caller, rest
; remains in list.  Remainder needs its size and next 
; fields set up, and the previous block next field 
; needs to be pointed at the remainder.  The callers part
; needs its size field set up too.  For brevitys sake in the
; comments below, n = request size, k = block size - n,
; The address of the new free block is x+n.
;

	sta	<$temp0		; save k.
	lda	13,s		; get n.
	sta	<$size,x	; size of callers block = n.
;
; Replace callers stacked size request
; with address of his block.
;
	txa			; get pointer to block.
	inc	a		; bump past size field.
	inc	a
	sta	13,s		; save it for return.
;
; Calculate addr of block remainder to
; be linked into the list.
;
	clc
	txa			; get addr beginning of block.
	adc	<$size,x	; add size given to caller.
	sta	<$temp1		; save addr of remainder.
;
; Point remainder block at next block,
; and set size field in remainder,
; point previous block at remainder.
;
	lda	<$next,x	; get addr next block.
	ldx	<$temp1		; get addr remainder.
	sta	<$next,x	; point remainder to next.
	lda	<$temp0		; get remainder size.
	sta	<$size,x	; save in remainder size field.
	stx	<$next,y	; next field of prev = addr remainder.
$done:

;	plb

	pld
	ply
	plx
	pla
	plp
	rtl

$extend:
;
; End of list reached.  If possible, extend the
; heap to satisfy callers request, else set callers
; carry to indicate failure and return.
; See if last free block at end of heap.
; If so, extend it, else extend the heap by the
; requested amount and allocate the new heap area.
; In the former case, we have to go back and search
; the free list for the predecessor of the last block
; in order to set its next field to nil.  In the latter case
; the pointer is already nil and should stay that way since
; the whole of the new block is being allocated.
;
	tyx			; point x at last free block.
	clc
	txa			; get addr last block.
	adc	<$size,x	; add its size - 1 to get highest
	dec	a		; addr in block.
	cmp	>0,HeapTop		; result = highest heap addr ?
	beq	$exlast		; br if yes, try extend last block.
;
; Make new block if possible.
;
	lda	>0,HeapTop		; get high heap addr.
	inc	a		; point to (potential) new block.
	tax			; save new block addr in x.
	clc
	adc	13,s		; add new size to addr.
	sta	<$temp0		; save (potential) HeapTop + 1.
	adc	>0,StkMin		; add min stack size.
	cmp	>0,StkBas		; result > current stack base ?
	bcc	$1		; br if no.
	brl	$error		; br if yes, we failed.
$1:
	lda	<$temp0		; success, get new HeapTop.
	dec	a
	sta	>0,HeapTop		; put it where it belongs.
	lda	13,s		; get size new block.
	sta	<$size,x	; init size field new block.
	txa			; make pointer for caller.
	inc	a
	inc	a
	sta	13,s		; save callers pointer.
	brl	$done		; done at last.

$exlast:

;
; Extend the last block to make it as
; large as the requested block, if possible.
;
	sec			; find out how much more heap needed.
	lda	13,s		; get size needed.
	sbc	<$size,x	; subtract avail amount.	
	clc			; add extra heap needed to current top.
	adc	>0,HeapTop
	sta	<$temp0		; save addr new HeapTop.
;
; Now we know where the top of the extended heap would
; be, see if the space between it and current stack base
; is large enough (>= StackMin).
;
	clc			; see if high heap + stack min > stk base.
	adc	>0,StkMin		; high heap + stack min.
	cmp	>0,StkBas		; enough room for stack left ?
	bcs	$error		; br if no.
	lda	<$temp0		; yes, get new heap top.
	sta	>0,HeapTop		; make it official.
	lda	13,s		; get requested size.
	sta	<$size,x	; extend last block to needed size.
;
; Even though we know where the block is, we need to
; go through the search again to find its predecessor.
; Set up for the search and go do it.
;
	ldy	##0
	ldx	<$next
	brl	$srch

$error:	
	lda	9,s		; get psw.
	ora	##1		; set carry bit.
	sta	9,s		; save callers psw.
	brl	$done		; clean up stack and rtl.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; FreHeap :					;
; Enter with pointer to block (from GetHeap)	;
; on stack.  Returns with stack clean.		;
; It is a ghastly error to free storage		;
; not acquired from GetHeap.			;
; Calling sequence :				;
;						;
; 	push pointer returned by GetHeap.	;
;	jsl	>0,FreHeap			;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; This routine is good about merging adjacent 
; free blocks, but doesn't try to shrink an
; extended heap.  It probably should.
; 
freheap:

$size equ 0			; size offset in block.
$next equ 2			; next ptr offset in block.
$temp0 equ 4			; temp area in head block.
$temp1 equ 6			; ditto

	php
	rep	#0x30
	pha
	phx
	phy
	phd

;	phb
;
; Set dbr and dpr = 0
;
;	pea	##0
;	plb
;	phb
;	pld

	lda	##0
	tcd
;
; set <temp0 = addr of block to free
; and <temp1 = temp0 + size of block,
; the address immediately following the
; block to be freed.
;
	lda	13,s		; get block addr + 2.
	dec	a		; point back to beginning.
	dec	a
	sta	<$temp0		; save block addr for later.
	tax			; point x at block.
	clc
	adc	<$size,x	; add size of block to addr.
	sta	<$temp1		; save for later.
;
; Search the free list for the blocks
; before and after (<temp0).
; Since the free list is sorted by block
; address (lowest first), we just look through
; it until we find an x such that next(x) >=  <temp1.
;
	lda	##0		; start at head of list.
$srch:	
	tax			; point x at current block.
	lda	<$next,x	; point a at next block.
	cmp	<$temp1		; next before one to free ?
	bcc	$srch		; br if yes, keep looking.
;
; The block to free (<temp0) is between 2 others,
; or is at the end of the list (acc = -1).
; x points to the lower one, acc to the higher.
; If the higher block is adjacent to (<temp0)
; merge the two.
;
	beq	$merge		; br if higher is adjacent.
;
; Not adjacent.  Set next(<temp0) = addr higher block
; and go check the lower block for adjacency.
;
	ldy	<$temp0		; point y at block to free.
	sta	$next,y		; point new free block to higher.
	bra	$chkb4		; go see if lower is adjacent.
;
; Higher block adjacent to (<temp0).
; Add size of higher block to size(<temp0),
; and copy next(higher) to next(<temp0).
;

$merge:
	stx	<$temp1		; save addr lower block. (acc = temp1)
	ldx	<$temp0		; point x at block to free.
	tay			; point y at higher block.
	clc
	lda	$size,y		; get highers size.
	adc	<$size,x	; add to freed size.
	sta	<$size,x	; save new freed size.
	lda	$next,y		; get highers next.
	sta	<$next,x	; save new freed next.
	ldx	<$temp1		; point x back at lower block.

;
; Do as above for the lower block - if its adjacent,
; merge, else just set next(lower) = <temp0.
;

$chkb4:
	clc			; calculate end of lower + 1.
	txa			; get addr lower in acc.
	adc	<$size,x	; add size(lower).
	cmp	<$temp0		; = start of block to free ?
	beq	$merg		; br if yes, merge'm.
;
; Not adjacent , point lower to freed block.
;
	lda	<$temp0		; get addr freed block.
	sta	<$next,x	; save in next(lower).
	bra	$done		; clean up stack and rtl.
;
; Adjacent, add size(<temp0) to size(lower),
; set next(lower) = next(<temp0).
;

$merg:
	ldy	<$temp0		; point y at block to free.
	clc
	lda	$size,y		; get freed block size.
	adc	<$size,x	; add to size(lower).
	sta	<$size,x	; save new lower size.
	lda	$next,y		; get next(freed).
	sta	<$next,x	; save new next(lower).

$done:

;
; Remove callers pointer from stack and rtl.
;

;	plb

	pld
	ply
	plx

; 7,s	pointer to block
; 5,s	rethi,bank
; 3,s	psw, retlo
; 1,s	acc

	lda	5,s
	sta	7,s
	lda	3,s
	sta	5,s
	lda	1,s
	sta	3,s

	pla
	pla
	plp
	rtl

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; ChkHeap 						;
; Go through the heap free list, looking for obvious	;
; signs of corruption, i.e. free blocks of preposterous	;
; size, pointers outside of the heap area etc.		;
; Return with carry clear if everything seems ok.	;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

chkheap:
 
$size equ 0			; size offset in block.
$next equ 2			; next ptr offset in block.
$temp0 equ 4			; temp area in head block.
$temp1 equ 6			; ditto

	clc
	php
	rep	#0x30
	pha
	phx
	phy
	phd
	phb
;
; set dbr = dpr = 0.
;
	pea	##0
	plb
	phb
	pld
;
; Pass 1 - check size word in all blocks
; (free and allocated), verify sum = Heaptop+1.
;

	lda	##8		; get pointer to first block.
$sizes:
	tax			; see if out of heap area.
	cmp	HeapTop
	bcs	$sized		; br if yes.
	adc	<$size,x	; add size, point to next block.
	bcc	$sizes		; should branch.
	bra	$error		; br if size ridiculous.
$sized:
	dec	a
	cmp	HeapTop
	bne	$error
;
; Pass 2 - verify that free list pointers
; are in the range 8 .. HeapTop - 3.
;
	lda	HeapTop
	dec	a
	dec	a
	sta	<$temp0

	ldx	##0		; point to head.
$range:
	lda	<$next,x	; get addr next free block.
	tax			; save for next iteration.
	bpl	$r1
	cmp	##0xffff	; end of list ?
	bne	$error		; error if minus but not ffff.
$r1:	cmp	##8		; addr < 8 ?
	bcc	$error		; br if yes, too small.
	cmp	<$temp0		; addr < HeapTop - 2 ?
	bcc	$range		; br if yes, its ok
	bra	$error		; nope,  points out of heap.
;
; Pass 3 - verify list is in address order.
; i.e. for each free block at p, verify
; that p+size < next.
; 
	ldx	##0		; point to head.
$ptrs:
	lda	<$next,x	; get ptr to next free block.
	bmi	$done		; br if end of list.
	tax			; ptr to x.
	clc			; add size of block to its addr.
	adc	<$size,x
	bcs	$error
	cmp	<$next,x	; result < addr next block ?
	bcc	$ptrs		; br if yes, check next.
				; no, fall into error.

$error:
	lda	10,s		; set carry bit in
	ora	##1		; callers psw and return.
	sta	10,s

$done:
	plb
	pld
	ply
	plx
	pla
	plp
	rtl
endheap:
	end
